/* Copyright (C) 2002-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef __VNCPARAMETERS_H__
#define __VNCPARAMETERS_H__

/**
 * \file vncparameters.h
 *
 * This header file contains macros that expand to the names of VNC viewer
 * parameters.  These are for use in calls to VNCViewerSetParameter() and
 * VNCViewerGetParameter().
 *
 * All parameter values are passed to and from the SDK as strings.  If a
 * parameter is listed as having an integer value, then the string should
 * contain the value formatted in decimal.  If a parameter is listed as having
 * a Boolean value, the string should have the value "0" or "1".
 *
 * \see VNCViewerSetParameter, VNCViewerGetParameter
 */

/**
 * \brief The RSA key pair for use with the session.
 *
 * If you think that the VNC server might require the VNC viewer to
 * authenticate itself, or that it might require an encrypted session, you
 * should assign a value to this parameter before calling VNCViewerConnect() or
 * VNCViewerListen().  Failing to do this will cause connections to any such
 * server to fail with status VNCViewerStatusRSAKeysRequired.
 *
 * An RSA key pair can be generated by calling VNCGenerateRSAKeys().
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * No default.
 *
 * \see VNCGenerateRSAKeys()
 */
#define VNCParameterRSAKeys "RSAKeys"

/**
 * \brief If true, the server will continue to share its display with other
 * viewers.
 *
 * If false, then the server will disconnect other viewers when you
 * successfully establish a connection.  However, the server will continue to
 * accept further connections.
 *
 * Not all VNC servers support multiple simultaneous viewers.  If connecting
 * to a server that supports only a single viewer, such as those provided as
 * part of VNC Mobile Solution, then VNCParameterShared has no effect.
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * Boolean, default true.
 */
#define VNCParameterShared "Shared"

/**
 * \brief A list of preferred RFB encodings.
 *
 * Setting this parameter causes a SetEncodings RFB message to be sent to the
 * VNC Server.  The message advertises all of the SDK's supported encodings to
 * the server, but indicates that the given encodings are the preferred ones.
 *
 * The default value for this parameter is "JPEG".  JPEG is a lossy encoding,
 * but requires much lower bandwidth than lossless encodings.  Under higher
 * bandwidth conditions (e.g. USB, Wi-Fi), you may prefer to use a lossless
 * encoding, such as ZRLE or TRLE, instead.
 *
 * Allowed values are "JPEG", "JRLE", "ZRLE", "TRLE", "SLRLE", "VAH264", 
 * "H264" and "Raw". String constants for use with VNCParameterPreferredEncoding
 * can be obtained with VNCGetEncodingName().
 *
 * Multiple values can be provided, separated by commas (for example,
 * "Raw,JPEG,ZRLE"), in decreasing order of preference.
 *
 * A VNC Server will usually honor the viewer's preferred encoding, provided
 * that that encoding is supported by the server.  However, the server may
 * choose to send updates using one of the other advertised encodings, if the
 * server considers this a more appropriate use of the available resources.
 *
 * The SDK supports a few other encodings for interoperability with legacy VNC
 * Servers, and will handle these encodings correctly if the server sends them.
 * However, it is not legal to select these as the preferred encoding.
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * The value is either "JPEG", "JRLE", "ZRLE", "TRLE", "SLRLE", "VAH264",
 * "H264" or "Raw", or some combination thereof (separated by commas).
 *
 * Default "JPEG".
 *
 * \see VNCViewerSetParameter(), VNCViewerGetParameter(), VNCParameterAdvertiseOnlyPreferredEncodings
 */
#define VNCParameterPreferredEncoding "PreferredEncoding"

/**
 * \brief Selects a particular desktop of a multi-desktop server.
 *
 * The value should be the desktop name or number, as it is known to the
 * server.
 *
 * Not all VNC servers support multiple desktops.  If connecting to a server
 * that supports only a single desktop, such as those provided as part of VNC
 * Mobile Solution, then VNCParameterSelectDesktop has no effect.
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * No default.
 *
 * \see VNCViewerSetParameter(), VNCViewerGetParameter()
 */
#define VNCParameterSelectDesktop "SelectDesktop"

/**
 * \brief If true, the SDK will send RFB KeyEvent messages to the server.
 *
 * If false, all KeyEvent messages will be suppressed, including those
 * generated by VNCViewerSendKeyEvent().
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * Boolean, default true.
 *
 * \see VNCParameterSendPointerEvents, VNCViewerSendKeyEvent()
 */
#define VNCParameterSendKeyEvents "SendKeyEvents"

/**
 * \brief If true, the SDK will send RFB PointerEvent messages to the server.
 *
 * If false, all PointerEvent messages will be suppressed, including those
 * generated by VNCViewerSendPointerEvent().
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * Boolean, default true.
 *
 * \see VNCParameterSendKeyEvents, VNCViewerSendPointerEvent()
 */
#define VNCParameterSendPointerEvents "SendPointerEvents"

/**
 * \brief Controls the SDK's logging.
 *
 * If the value is an empty string, then all logging is disabled.  If the
 * string is non-empty, then it is a comma-separated list of directives of the
 * form category:severity.
 *
 * category may be "*", which means that the selected severity is to be used
 * for all log categories.
 *
 * severity is an integer from 0 to 100 inclusive.  The SDK defines these
 * logging severities, but may use other values in between:
 *
 *  -   0 - serious errors
 *  -  10 - warning and connection status messages
 *  -  30 - informational messages
 *  - 100 - verbose debugging
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * Default "" (logging is disabled).
 *
 * \see VNCViewerLogCallback
 */
#define VNCParameterLog "Log"

/**
 * \brief Controls the behavior of RA2Rev and RA2Revne security types.
 *
 * When a server connects to the viewer SDK with either the RA2Rev or the
 * RA2Revne security type, then if this parameter is true the server will
 * be prompted to enter a password.  This password is passed to your
 * application's VNCViewerAsynchronousValidatePasswordCallback() for
 * verification.
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * Boolean, default false.
 *
 * \see VNCViewerAsynchronousValidatePasswordCallback
 */
#define VNCParameterAuthenticateServer "AuthenticateServer"

/**
 * \brief The alignment of the server display relative to the skin image, if
 * a mobile device skin does not define a display rectangle.
 *
 * Each skinface image within a mobile device skin typically defines a display
 * rectangle.  This is the 'window' within the skinface image into which the
 * Viewer SDK will render the server display.
 *
 * In the case that a mobile device skin does not define a display rectangle,
 * the position of the server display is determined by the value of this
 * parameter.  The legal values for this parameter are:
 *
 *  - "top" - the server display will be rendered above the skinface image
 *  - "bottom" - the server display will be rendered beneath the skinface image
 *  - "left" - the server display will be rendered to the left of the skinface
 *    image
 *  - "right" - the server display will be rendered to the right of the
 *    skinface image
 *
 * The value is case-insensitive.
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * Default "right".
 *
 * \see VNCViewerSetSkin()
 */
#define VNCParameterSkinAlignment "SkinAlignment"

/**
 * \brief The background color used to fill transparent areas in images in
 * mobile device skins.
 *
 * Skinface images for mobile devices will typically contain areas that are
 * either fully or partially transparent.  The Viewer SDK will automatically
 * alpha-blend these areas with the pixel value given by the
 * SkinBackgroundColor.
 * 
 * The value is a string that must be six characters long and composed of
 * hexadecimal digits.  It as interpreted as a three-byte RGB colour value, as
 * in HTML (so "000000" is black and "ffffff" is white).  (Transparent or
 * alpha-blended skin backgrounds are not supported).
 *
 * Changes to this parameter will not take effect until the next call to
 * VNCViewerSetSkin().  This is because the alpha-blending is performed at the
 * time that the skinface images are decoded.
 *
 * Default "ffffff" (i.e. white).
 *
 * \see VNCViewerSetSkin()
 */
#define VNCParameterSkinBackgroundColor "SkinBackgroundColor"

/**
 * \brief The default highlight color used to highlight skin buttons over which
 * the user is hovering the mouse.
 *
 * Some skins specify a hover highlight color from within their skin XML.  This
 * value is used for skins that do not do so.
 *
 * The value is a string that must be eight characters long and composed of
 * hexadecimal digits.  It is interpreted as a four-byte RGBA colour value.
 * The first three bytes are the colour, and the fourth byte is an alpha value
 * (from 0, meaning fully transparent, to ff, meaning fully opaque, inclusive).
 *
 * Default "80808080" (i.e. gray with 50% transparency).
 *
 * \see VNCViewerSetSkin(), VNCParameterSkinButtonPressHighlight
 */
#define VNCParameterSkinButtonHoverHighlight "SkinButtonHoverHighlight"

/**
 * \brief The default highlight color used to highlight skin buttons over which
 * the user is pressing the mouse.
 *
 * Some skins specify a press highlight color from within their skin XML.  This
 * value is used for skins that do not do so.
 *
 * The value is a string that must be eight characters long and composed of
 * hexadecimal digits.  It is interpreted as a four-byte RGBA colour value.
 * The first three bytes are the colour, and the fourth byte is an alpha value
 * (from 0, meaning fully transparent, to ff, meaning fully opaque, inclusive).
 *
 * Default "ffffff80" (i.e. white with 50% transparency).
 *
 * \see VNCViewerSetSkin(), VNCParameterSkinButtonPressHighlight
 */
#define VNCParameterSkinButtonPressHighlight "SkinButtonPressHighlight"

/**
 * \brief The DTD that is used to validate XML contained in mobile device skins.
 *
 * Prior to loading a mobile device skin, the value of this parameter is parsed
 * as an XML DTD.  The parsed DTD is then used to validate the skin XML
 * contained in the mobile device skin.
 *
 * The DTD is configurable in order to allow forward compability with skins
 * developed for future releases of the Viewer SDK.
 *
 * Note that the value of this parameter is the DTD text itself, and not a
 * filename.
 *
 * Default is the skin XML DTD for VNC Mobile Solution 1.1.
 * 
 * \see VNCViewerSetSkin()
 */
#define VNCParameterSkinDTD "SkinDTD"

/**
 * \brief The delay in milliseconds between key down events being injected by
 * the SDK API and their being sent to the VNC Server.
 * 
 * This parameter governs a workaround for spurious key auto-repeats generated
 * by VNC Servers being accessed over networks with high-latency (e.g. GPRS, or
 * even the Internet, at long distance).
 *
 * When the value of this parameter is non-zero, a key down event injected
 * through the SDK API causes a timer to be set with a delay in milliseconds
 * equal to the value of the VNCParameterKeyDownDelay parameter.  When the
 * timer expires, all pending key down events are sent to the server
 * immediately.  If a key is released before the timer expires, all pending key
 * down events and the new key up event are sent to the server immediately, and
 * the timer is cancelled.
 *
 * The effect of this is that, when the local user is typing normally, the key
 * down event and key up event for each short key press are coalesced into a
 * single network packet.  This will very likely result in the VNC Server
 * receiving them both at the same time.  Without the workaround, a
 * high-latency network might cause the server to receive the key up event
 * significantly later than the key down event, causing the server host OS to
 * auto-repeat the key, even if that was not the user's intention.
 *
 * If you know that a connection will always be made over a network with low
 * latency (e.g. a wired or Wi-Fi network), or if you just prefer to disable
 * this workaround, then you can safely set this parameter to 0.  This will
 * cause key down events to be send to the server immediately after they are
 * injected into the SDK.
 *
 * Changes to this parameter will take effect even if the viewer thread is
 * already running.
 *
 * Integer, default 375 (milliseconds), minimum 0.
 *
 * \see VNCViewerSendKeyEvent(), VNCViewerSendDeviceKeyEvent(),
 * VNCViewerSendDeviceKeyOrKeyEvent()
 */
#define VNCParameterKeyDownDelay "KeyDownDelay"

/**
 * \brief The directory in the local filesystem from which the SDK will attempt
 * to load bearer DLLs and shared objects.
 *
 * Whenever a VNC command string is processed, the SDK attempts to load the
 * bearer for the session from a DLL or shared object.  The filename is
 * 'vncbearer-TYPE.dll' on Windows, 'libvncbearer-TYPE.dylib' on Mac OS X,  or
 * 'libvncbearer-TYPE.so' on UNIX, where TYPE is the value of the 't' field in
 * the command string.  The directory in which the DLL or shared object is
 * expected to exist is given by the value of the VNCParameterBearerDirectory
 * parameter.
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * An empty value indicates that the system library search method should be
 * used.
 * 
 * A value of "." will use the current working directory.
 *
 * Default "" (system library search method).
 */
#define VNCParameterBearerDirectory "BearerDirectory"

/**
 * \brief The quality setting that the VNC Server will use when sending
 * updates using the JPEG RFB encoder.
 *
 * JPEG is an efficient but lossy encoding.  Use of JPEG with VNC is
 * recommended in cases where image fidelity is of secondary importance to
 * performance.
 *
 * Using JPEG gives much better performance against some devices. Also,
 * because of its lossiness, JPEG also produces smaller compressed data
 * than ZRLE, which can help with devices that only have access to a
 * GPRS cellular network.
 *
 * Valid values are in the range 0 to 100 inclusive, with higher numbers
 * preserving more detail at the cost of using more bandwidth.  Sensible values
 * are in the range 50 to 100, although text is usually still legible with
 * lower quality settings.
 *
 * For more information on how this setting affects the quality of images
 * encoded with JPEG, please see
 * http://en.wikipedia.org/wiki/JPEG#Effects_of_JPEG_compression.
 *
 * Note that all JPEG images contain 24-bit RGB888 pixel data.  For this
 * reason, the session pixel format chosen with VNCViewerSetPixelFormat() is
 * ignored by VNC Servers when sending JPEG-encoded updates.
 *
 * Integer, default 75, minimum 0, maximum 100.
 *
 * \see VNCParameterPreferredEncoding, VNCViewerSetPixelFormat()
 */
#define VNCParameterEncodingJPEGQuality "EncodingJPEGQuality"

/**
 * \brief The size to limit cut text to before sending to the VNC Server.
 *
 * It is generally undesirable to send large cut text to the VNC server due
 * to the time taken to transfer the data as well as memory usage
 * considerations within the receiving VNC server. By truncating the cut
 * text to an upper limit of bytes before sending then the adverse performance
 * impact of large blocks of cut text can be avoided.
 *
 * If the value is non-zero, then a single call to VNCViewerSendClientCutText()
 * will never send more than this number of UTF-8 characters to the VNC Server.
 * The remaining text is silently discarded.
 * 
 * If the value is zero, then VNCViewerSendClientCutText() will always send all
 * of the text to the VNC Server.
 *
 * Integer, default 16384 (characters), minimum 0.
 *
 * \see VNCViewerSendClientCutText()
 */
#define VNCParameterMaxCutText "MaxCutText"

/**
 * \brief The major component of the version number of the earliest revision of
 * the RFB protocol that the viewer will allow.
 *
 * The supported revisions of the RFB protocol are 4.1, 3.8 and 3.7.  Upon
 * connection to a VNC Server that does not offer one of these protocol
 * revisions, the Viewer SDK will disconnect and report the error
 * VNCViewerProtocolMismatch.
 *
 * You can further restrict the protocol revisions by setting
 * VNCParameterMinimumRFBVersionMajor and VNCParameterMinimumRFBVersionMinor
 * appropriately.  For example, setting VNCParameterMinimumRFBVersionMajor to 4
 * and VNCParameterMinimumRFBVersionMajor to 0 will disallow RFB 3.7 and 3.8
 * sessions.
 *
 * By default, VNCParameterMinimumRFBVersionMajor and
 * VNCParameterMinimumRFBVersionMinor are both 0, so that any supported
 * revision of the protocol is supported.
 *
 * Integer, default 0, minimum 0.
 *
 * \see VNCParameterMinimumRFBVersionMinor,
 * VNCViewerPropertyRFBVersionMajor, VNCViewerPropertyRFBVersionMinor
 */
#define VNCParameterMinimumRFBVersionMajor "MinimumRFBVersionMajor"

/**
 * \brief The minor component of the version number of the earliest revision of
 * the RFB protocol that the viewer will allow.
 *
 * The supported revisions of the RFB protocol are 4.1, 3.8 and 3.7.  Upon
 * connection to a VNC Server that does not offer one of these protocol
 * revisions, the Viewer SDK will disconnect and report the error
 * VNCViewerProtocolMismatch.
 *
 * You can further restrict the protocol revisions by setting
 * VNCParameterMinimumRFBVersionMajor and VNCParameterMinimumRFBVersionMinor
 * appropriately.  For example, setting VNCParameterMinimumRFBVersionMajor to 4
 * and VNCParameterMinimumRFBVersionMajor to 0 will disallow RFB 3.7 and 3.8
 * sessions.
 *
 * By default, VNCParameterMinimumRFBVersionMajor and
 * VNCParameterMinimumRFBVersionMinor are both 0, so that any supported
 * revision of the protocol is supported.
 *
 * Integer, default 0, minimum 0.
 *
 * \see VNCParameterMinimumRFBVersionMajor,
 * VNCViewerPropertyRFBVersionMajor, VNCViewerPropertyRFBVersionMinor
 */
#define VNCParameterMinimumRFBVersionMinor "MinimumRFBVersionMinor"

/**
 * \brief The time interval in milliseconds with which the Viewer SDK will
 * rate-limit pointer and touch events that are sent to the server.
 *
 * Normally, one RFB PointerEvent message is sent to the server for
 * each invocation of VNCViewerSendPointerEvent().  Similarily one RFB
 * TouchEvent message is sent to the server for each invocation of
 * VNCViewerSendTouchEvent().  If the server host reacts to each
 * pointer or touch event by redrawing a large area of the screen (e.g. if a
 * mouse drag causes smooth scrolling), then performance can degrade
 * rapidly.  Limiting the rate with which the SDK sends RFB
 * PointerEvent and TouchEvent messages to the server can often lead to a large
 * increase in perceived performance over a low-bandwidth network.
 *
 * If VNCParameterPointerRateLimit is non-zero, then it is the minimum
 * interval in milliseconds between RFB PointerEvent and TouchEvent
 * messages that the SDK generates as a result of calls to
 * VNCViewerSendPointerEvent() and VNCViewerSendTouchEvent().  (This
 * remains true as long as the button state does not change; if a call
 * to VNCViewerSendPointerEvent() supplies a VNCPointerDeviceButton
 * mask that is different from the VNCPointerDeviceButton mask
 * supplied to the previous call, then the new pointer event is sent
 * immediately, and the rate-limit timeout is reset.  Similarily, if
 * the number of touch identifiers or button mask in a touch event
 * changes then the new touch event is sent immediately)
 *
 * Some VNC Servers may request pointer and touch events rate-limiting
 * as part of feature negotiation.  The Viewer SDK will rate-limit
 * pointer and touch events either if the server requests it or if
 * this parameter is set to a non-zero value.  If both the SDK API and
 * the server request pointer and touch rate limiting, then the
 * timeout used is the one specified by the API (i.e. the value of
 * VNCParameterPointerRateLimit).
 *
 * Integer, default 0 (no pointer or touch rate limiting), minimum 0.
 *
 * \see VNCViewerSendPointerEvent(), VNCPointerDeviceButton
 * \see VNCViewerSendTouchEvent(), VNCTouchDescriptor.identifier,
 * \see VNCTouchDescriptor.buttonMask
 */
#define VNCParameterPointerRateLimit "PointerRateLimit"

/** 
 * \brief Enables MirrorLink.
 * 
 * If this VNCParameterMirrorLink is true, then any RFB 3 sessions are 
 * treated as MirrorLink sessions.  Otherwise, they are treated as plain RFB 
 * 3 sessions. 
 * 
 * Successful operation of MirrorLink requires a MirrorLink-compatible 
 * VNC Server.  It is necessary to set this parameter to true in advance in 
 * order for MirrorLink sessions to work correctly. 
 * 
 * Note that setting this parameter to true and then connecting to a regular 
 * RFB 3 server will *not* work.  It is necessary to know in advance whether 
 * RFB 3 servers are expected to be MirrorLink servers. 
 * 
 * This parameter has no effect on RFB 4 sessions. 
 * 
 * Changes to this parameter after the viewer thread has started will have no 
 * effect. 
 * 
 * Boolean, default false. 
 */ 
#define VNCParameterMirrorLink "MirrorLink"

/**
 * \brief Enables MirrorLink High Speed Media Link (HSML).
 *
 * If this parameter is true, then the SDK shall advertise support for HSML
 * in MirrorLink sessions. If the MirrorLink server also supports HSML, then
 * your application shall receive the VNCViewerHSMLInformationReceivedCallback
 * each time HSML information is received from the server.
 *
 * Note that your application must establish the HSML connection separately.
 * The VNC Viewer SDK only provides the functionality required on the RFB
 * connection in order to support the HSML session. The VNC HSML SDK may be
 * used to establish and maintain an HSML connection.
 *
 * For more information, see \ref mirrorlink_hsml.
 *
 * This parameter has no effect on non-MirrorLink sessions.
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * Boolean, default false.
 *
 * \see VNCViewerHSMLInformationCallback
 */
#define VNCParameterHSML "HSML"

/**
 * \brief The time interval in milliseconds that must elapse before the Viewer
 * SDK will convert a pointer event from a 'touch' to a 'press'.
 *
 * This feature is intended for use when using a viewer host with a 
 * single-touch touchscreen to control a touchscreen device that has a UI
 * system that distinguishes between being the screen being touched and the
 * screen being pressed.  VNC Servers on such devices interpret 
 * VNCPointerDeviceButton1 as a touch and VNCPointerDeviceButton2 as a press.
 *
 * If touch-to-press conversion is enabled, then short clicks at the viewer
 * become touches at the server, and long clicks at the viewer become presses
 * at the server.  The resulting user experience is very similar to that when
 * using the device itself.
 *
 * You may wish to disable touch-to-press conversion if you prefer to have your
 * application offer some other means for the user to send press events to a
 * server that requires them.  For example, your application could offer a
 * second mouse button, or a multi-touch or pressure-sensitive touchscreen.
 *
 * Specifically, when touch-to-press conversion is enabled, the events injected
 * by calls to VNCViewerSendPointerEvent() are modified as follows:
 *
 *  - short clicks of VNCPointerDeviceButton1 are sent to the server unmodified
 *  - long clicks of VNCPointerDeviceButton1 are converted to clicks of
 *    VNCPointerDeviceButton2 (i.e. what would have been a touch is converted
 *    to a press)
 *  - drags of VNCPointerDeviceButton1 are sent to the server unmodified
 *
 * (The duration of a click is given by the elapsed time between the first call
 * to VNCViewerSendPointerEvent() that includes that button in the mask and the
 * next call to VNCViewerSendPointerEvent() that does not include that button
 * in the mask.  A drag occurs when the pointer location changes significantly
 * during a click; see \ref VNCParameterTouchPressDragDistance for further
 * information.)
 *
 * If VNCParameterTouchPressTimeout is non-zero, then touch-to-press conversion
 * will be enabled if it is requested by the server, and
 * VNCParameterTouchPressTimeout gives the minimum duration of a long click.
 *
 * If VNCParameterTouchPressTimeout is zero, then touch-to-press conversion is
 * always disabled, even if the VNC Server requests it.
 *
 * Integer, default 250 (250ms before a touch becomes a press), minimum 0.
 *
 * \see VNCViewerSendPointerEvent(), VNCPointerDeviceButton,
 * VNCParameterTouchPressDragDistance
 */
#define VNCParameterTouchPressTimeout "TouchPressTimeout"

/**
 * \brief The number of pixels difference in pointer events before the Viewer
 * SDK will not convert a touch event to a press event.
 *
 * This parameter is used in conjunction with
 * \ref VNCParameterTouchPressTimeout
 * to configure touch-to-press conversion in the Viewer SDK.
 *
 * If VNCParameterTouchPressDragDistance is non-zero, and touch-to-press
 * conversion is enabled, then VNCParameterTouchPressDragDistance gives the
 * minimum horizontal or vertical displacement from the initial click location
 * before a click is considered a drag.  This is important because drags are
 * never converted to presses, but a click may be converted to a press if it
 * lasts long enough.
 *
 * If VNCParameterTouchPressDragDistance is zero, then clicks are never
 * considered to be drags.
 *
 * This parameter has no effect if touch-to-press conversion is disabled.
 *
 * Integer, default 3 (3 pixel movement), minimum 0.
 *
 * \see VNCViewerSendPointerEvent(), VNCPointerDeviceButton,
 * VNCParameterTouchPressTimeout
 */
#define VNCParameterTouchPressDragDistance "TouchPressDragDistance"

/**
 * \brief Enable measuring of specified performance metrics.
 *
 * This parameter allows the performance of a live VNC session to be measured.
 * The following performance metrics are supported:
 *
 *  - FramesRendered: Frames Rendered per Second
 *  - PixelThroughput: Pixel Throughput (pixels per second)
 *  - CallbackCompletion: Callback Completion Time (milliseconds)
 *
 * Metrics are measured over consecutive sample periods, whose duration is 
 * defined by \ref VNCParameterPerformanceMeasuringSampleDuration. At the 
 * end of each sample period, performance results are reported through 
 * \ref VNCViewerPerformanceResultCallback.
 *
 * The FramesRendered metric is dependent on receiving notifications from the 
 * viewer application when it has rendered its image of the framebuffer screen.
 * The viewer application provides this notification through 
 * \ref VNCViewerFrameBufferRendered. If this notification is not provided, 
 * then this metric will consistently report a value of zero.
 *
 * \note The SDK will not be able to measure the frames rendered per second if
 * a pluggable renderer is being used which does not call the
 * \ref VNCRendererOnFrameRendered suppport API. In this case, the
 * FramesRendered metric will consistently report a value of zero. Pluggable
 * renderers may choose to independently report frame rate information through
 * some channel (such as the SDK's log).
 *
 * The CallbackCompletion metric measures the mean completion time over the
 * sample period for the following callback functions:
 *
 *  - \ref VNCViewerFrameBufferUpdateStartCallback
 *  - \ref VNCViewerFrameBufferUpdateEndExCallback
 *  - \ref VNCViewerLockRectangleExCallback
 *  - \ref VNCViewerUnlockRectangleExCallback
 *  - \ref VNCViewerLogCallback
 *  - \ref VNCViewerDesktopResizeCallback
 *  - \ref VNCViewerServerCutTextCallback
 *  - \ref VNCViewerExtensionEnabledCallback
 *  - \ref VNCViewerExtensionMessageReceivedCallback
 * 
 * The total time spent in each callback function during the sample period shall
 * also be reported, as a percentage of the sample period's duration.
 *
 * Under the WindowsCE platform the CallbackCompletion results are dependant 
 * upon a sufficiently accurate implementation of GetSystemTime() being present 
 * on the device. Should the granularity of this implementation be larger than 
 * the execution time for a callback then that call will not contribute to the 
 * results for the current sample. Should this happen for every instance of a 
 * particular callback then the average time spent per call will be reported 
 * as 0ms.
 *
 * The name of each metric to enable should be specified as a colon separated
 * list. To enable all performance metrics, set this parameter to 
 * "FramesRendered:PixelThroughput:CallbackCompletion". Setting this parameter 
 * to "" disables performance measuring.
 *
 * This parameter may be set at any time. If performance measuring is currently
 * active, any changes to this parameter shall take effect from the next sample
 * period.
 * 
 * String, default "" (performance measuring disabled)
 *
 * \see VNCParameterPerformanceMeasuringSampleDuration, VNCViewerPerformanceResultCallback,
 * VNCViewerFrameBufferRendered, VNCRendererOnFrameRendered
 */
#define VNCParameterPerformanceMeasuring "PerformanceMeasuring"

/**
 * \brief Specify the duration of the sample period to use when measuring performance.
 *
 * This parameter is to be used in conjunction with \ref VNCParameterPerformanceMeasuring.
 *
 * Specifies the duration, in milliseconds, of the sample period to use when 
 * measuring performance. All performance metrics are measured over a sample period,
 * and results are reported at the end of the sample period.
 *
 * This parameter may be set at any time. If performance measuring is currently
 * active, any changes to this parameter shall take effect from the next sample
 * period.
 *
 * Integer, default 5000 (5 seconds), minimum 1000 (1 second).
 *
 * \see VNCParameterPerformanceMeasuring.
 */
#define VNCParameterPerformanceMeasuringSampleDuration "PerformanceMeasuringSampleDuration"

/**
 * \brief Specify when automatic update requests should be sent.
 *
 * When requesting updates automatically the SDK can request the next
 * update at one of two points:
 *
 * -# As soon as the previous update starts. This corresponds to about the time
 *    when the VNCViewerFrameBufferUpdateStartCallback is called.
 * -# Once the previous update has been completely received. This corresponds to
 *    about the time when the VNCViewerFrameBufferUpdateEndExCallback is called.
 * 
 * Requesting the next update as soon as the previous update is received allows
 * the server to encode and send the next update while the SDK is still
 * waiting to receive or process the previous update. This can lead to an
 * improved update rate on some connection types.
 *
 * There are three main reasons why early updates might not be desirable:
 *
 * -# MirrorLink requires that all FramebufferBlockingNotification messages for
 *    content in a particular FramebufferUpdate are sent by the client before
 *    it sends the next FramebufferUpdateRequest.
 *
 * -# Some servers ignore update requests if they are received while the server
 *    is still sending an update. For these servers the screen will not update
 *    automatically if updates are automatically requested as soon as the
 *    previous update starts.
 *
 * -# If the SDK is running on a system which can't process and display updates
 *    as quickly as the server can send them. On a resource limited system
 *    requesting an update only at the end of the previous one forces the
 *    server to only send an update once the SDK has fully decoded all the
 *    data.
 *
 * A value of "UpdateStart" will cause the automatic request for the next
 * update to occur as soon as the previous update starts.
 * 
 * A value of "UpdateEnd" will cause the automatic request for the next update
 * to occur when the previous update has been completely received.
 *
 * A value of "Detect" will cause the automatic request to occur when the
 * previous update has been completely received unless the SDK is connected to
 * a server which can be reliably detected to support early update requests.
 *
 * "Detect" will never enable early update requests for MirrorLink servers.
 *
 * If automative update requests have been disabled via a call to
 * VNCViewerEnableDisplayUpdates(FALSE) then this parameter has no effect.
 *
 * The value is either "UpdateStart", "UpdateEnd" or "detect"
 *
 * Default "Detect"
 *
 * \see VNCViewerEnableDisplayUpdates
 */
#define VNCParameterRequestUpdatesEvent "RequestUpdatesEvent"

/**
 * \brief Specifies the invalid region tracking mode of operation.
 *
 * The Viewer SDK is able to track the regions that become invalid during a frame
 * buffer update and this can be used to improve the performance of the viewer
 * application. There are three modes of operation:
 *
 * -# no tracking done in the SDK (the default mode) - 
 *        VNCViewerInvalidRegionTrackingNone
 * -# the SDK tracks the rectangles that become invalidated and returns a bounding
 * region containing all the invalidated regions -
 *        VNCViewerInvalidRegionTrackingBounding
 * -# the SDK tracks regions that become invalidated and returns each of them -
 *        VNCViewerInvalidRegionTrackingFull
 *
 * The mode of operation needs to be set before calling VNCViewerConnect.
 * Setting the parameter afterwards will result in undefined behavior.
 *
 * It is recommended for the application to choose a mode of operation
 * different from the default one. In the default mode the responsibility to
 * keep track of invalid regions remains with the client application. This adds
 * complexity. Also, the SDK is able to keep track more efficiently of the
 * regions, and therefore allowing it to do so results in better overall
 * performance.
 *
 * If not in the default mode, the tracked regions will be given to the 
 * application during the VNCViewerFramBufferUpdateEndExtCallback. At that point
 * the application can redraw the regions specified by the callback.
 *
 * Setting the parameter to an option different from
 * VNCViewerInvalidRegionTrackingNone, VNCViewerInvalidRegionTrackingBounding,
 * or VNCViewerInvalidRegionTrackingFull, will result in an error and the
 * Viewer SDK mode of operation will remain unchanged.
 *
 * \see VNCViewerGetParameter, VNCViewerSetParameter,
 * VNCViewerFrameBufferUpdateEndExCallback,
 * VNCViewerInvalidRegionTrackingNone, VNCViewerInvalidRegionTrackingBounding,
 * VNCViewerInvalidRegionTrackingFull
 */
#define VNCParameterInvalidRegionTracking "InvalidRegionTracking"

/**
 * \brief Specifies the timeout for connection establishment.
 *
 * The value is the timeout in milliseconds that the SDK imposes on connection
 * establishment. This timeout is used for all sessions, regardless of which
 * bearer a session uses.
 *
 * If the value of this parameter is non-zero, then a timer is set when the SDK
 * instructs the bearer to establish the connection. If the timeout expires
 * before the bearer reports that the connection is established, then the SDK
 * terminates the session with VNCViewerErrorConnectionTimedOut.
 *
 * If the value of this parameter is zero, then the SDK does not impose a
 * timeout on connection establishment. The application can still cancel an
 * in-progress connection attempt with VNCViewerReset().
 *
 * Note that this timeout governs only the bearer's establishment of the
 * connection (i.e. transport-layer setup). This timeout does not govern RSA
 * key generation, the RFB handshake, the MirrorLink handshake, skin fetching,
 * etc.
 *
 * Note also that both the bearer and the operating system may impose their own
 * timeouts on any operation that the bearer performs during connection
 * establishment. For example, under Windows and Windows CE, WSAConnect() may
 * fail with WSAETIMEDOUT if the system-global TCP connection establishment
 * timeout elapses. The SDK reports VNCViewerErrorConnectionTimedOut in these
 * cases, too.
 *
 * A change to the value of this parameter after starting the viewer thread
 * does not take effect until the next time the viewer thread is started.
 *
 * Integer, default 0 (connection attempts do not timeout), minimum 0.
 *
 * \see VNCViewerConnect(), VNCViewerListen(), VNCViewerProcessCommandString()
 */
#define VNCParameterConnectionEstablishmentTimeout "ConnectionEstablishmentTimeout"

/**
 * \brief Specifies the interval between watchdog pings.
 *
 * Watchdog pings are used to detect connection loss by exchanging small
 * messages with the VNC Server. This is useful in cases where the bearer or
 * the operating system cannot be relied upon to detect connection loss. For
 * example, Linux will destroy the associated network interface when a USB
 * cable is unplugged, but this does not result in a socket error being
 * reported to a TCP-over-USB bearer.
 *
 * VNCParameterWatchdogPingInterval and ::VNCParameterWatchdogPingCount are
 * used to configure this feature. VNCParameterWatchdogPingInterval gives the
 * interval in milliseconds between the pings that the Viewer SDK sends to the
 * VNC Server. The SDK counts the number of pings that have been sent but not
 * to which a response has not yet been received. If this count exceeds
 * VNCParameterWatchdogPingCount, then the SDK ends the session and reports
 * ::VNCViewerErrorDisconnected to the application.
 *
 * The values configured with VNCParameterWatchdogPingCount and
 * VNCParameterWatchdogPingInterval are communicated to compatible VNC Servers,
 * so that connection loss can be simultaneously detected at both ends using
 * the same criteria.
 *
 * By default, VNCParameterWatchdogPingInterval is 10000 and
 * VNCParameterWatchdogPingCount is 6, so that disconnection is reported after
 * one minute of unresponsiveness. If the connection to the VNC Server is over
 * a fast, low-latency transport (e.g. USB), then smaller values may be more
 * appropriate.
 *
 * Setting either VNCParameterWatchdogPingInterval or
 * VNCParameterWatchdogPingCount to zero disables this feature entirely.
 *
 * Watchdog pings are not available in RFB 3 sessions (including MirrorLink
 * sessions).
 *
 * Integer, default 10000 (every 10 seconds), minimum 0.
 *
 * \see VNCParameterWatchdogPingCount
 */
#define VNCParameterWatchdogPingInterval "WatchdogPingInterval"

/**
 * \brief Specifies the maximum pending number of responses to watchdog pings
 * before the SDK will regard the connection as having been lost.
 *
 * See ::VNCParameterWatchdogPingInterval for an explanation of this parameter.
 *
 * Integer, default 6, minimum 0.
 *
 * \see VNCParameterWatchdogPingInterval
 */
#define VNCParameterWatchdogPingCount "WatchdogPingCount"

/**
 * \brief Specifies the linger time on disconnection.
 *
 * The value is the timeout in milliseconds that the SDK uses when
 * disconnecting. This timeout is used for all sessions, regardless of
 * which bearer a session uses.
 *
 * If the value of this parameter is non-zero, then a timer is set
 * when the SDK starts to close a connection. If the timeout expires
 * before any outstanding data has been sent, then the SDK terminates
 * the session with VNCViewerErrorConnectionTimedOut.
 *
 * If the value of this parameter is zero, then the SDK does not
 * impose a timeout on the sending of outstanding data on
 * disconnection. The SDK will wait until either there is no
 * outstanding data or an error occurs.
  *
 * Note that this timeout governs only the sending of outstanding data
 * on disconnection when no error is reported by the bearer. If the bearer
 * reports an error then the connection will be terminated immediately.
 *
 * Note also that both the bearer and the operating system may impose
 * their own timeouts on the closing of connections once the SDK has
 * sent any outstanding data. For example, under Windows, Windows CE
 * and Linux, the data sending behaviour of closing sockets can be
 * adjusted using the SO_LINGER and SO_DONTLINGER socket options.
 *
 * A change to the value of this parameter after starting the viewer thread
 * does not take effect until the next time the viewer thread is started.
 *
 * Integer, default 60000 (1 minute), minimum 0.
 *
 * \see VNCViewerConnect(), VNCViewerListen(), VNCViewerProcessCommandString()
 */
#define VNCParameterDisconnectionLingerTimeout "DisconnectionLingerTimeout"

/**
 * \brief Combination of flags that governs which factors in content attesation
 * are attested by a MirrorLink server.
 *
 * To enable content attestation, both
 * VNCParameterContentAttestationSignatureFlag and
 * ::VNCParameterMirrorLinkVNCServerPublicKey must be set to sensible values.
 *
 * See \ref mirrorlink_ca for a full explanation.
 * 
 * Integer, default 0, minimum 0.
 */
#define VNCParameterContentAttestationSignatureFlag "ContentAttestationSignatureFlag"

/**
 * \brief Base-64 encoded RSA public key of the MirrorLink
 * TerminalMode:VNC-Server component, as returned by the Device Attestation
 * Protocol.
 *
 * If you are using the VNC DAP SDK, then this the applicationPublicKeyBase64
 * string in the attestation for VNCDAPComponentTerminalModeVNCServer.
 *
 * To enable content attestation, both
 * ::VNCParameterContentAttestationSignatureFlag and
 * VNCParameterMirrorLinkVNCServerPublicKey must be set to sensible values.
 *
 * See \ref mirrorlink_ca for a full explanation.
 *
 * String, default "".
 */
#define VNCParameterMirrorLinkVNCServerPublicKey "MirrorLinkVNCServerPublicKey"

/**
 * \brief If true, the viewer will advertise that it supports only the
 * encodings specified in the list parameter "PreferredEncoding". Additional
 * non-pixel encoding values (pseudo-encodings) may also be advertised to
 * support protocol extensions.
 *
 * If false, the viewer will advertise all supported encodings, but
 * prefers those specified in the list parameter "PreferredEncoding".
 *
 * \see VNCParameterPreferredEncodings
 *
 * Boolean, default false.
 */
#define VNCParameterAdvertiseOnlyPreferredEncodings "AdvertiseOnlyPreferredEncodings"

/**
 * \brief The directory in the local filesystem from which the SDK will attempt
 * to load decoder DLLs and shared objects.
 *
 * Whenever VNCViewerLoadDecoderFromFile is called, the SDK attempts to load the
 * decoder from a DLL or shared object.  The filename is 'vncdecoder-NAME.dll'
 * on Windows, 'libvncdecoder-NAME.dylib' on Mac OS X, or
 * 'libvncdecoder-NAME.so' on UNIX, where NAME is the string provided to the API
 * call.  The directory in which the DLL or shared object is expected to exist
 * is given by the value of the VNCParameterDecoderDirectory parameter.
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * An empty value indicates that the system library search method should be
 * used.
 *
 * A value of "." will use the current working directory.
 *
 * Default "" (system library search method).
 */
#define VNCParameterDecoderDirectory "DecoderDirectory"

/**
 * \brief The directory in the local filesystem from which the SDK will attempt
 * to load renderer DLLs and shared objects.
 *
 * Whenever VNCViewerLoadRendererFromFile is called, the SDK attempts to load
 * the renderer from a DLL or shared object.  The filename is
 * 'vncrenderer-NAME.dll' on Windows, 'libvncrenderer-NAME.dylib' on Mac OS X,
 * or 'libvncrenderer-NAME.so' on UNIX, where NAME is the string provided to
 * the API call.  The directory in which the DLL or shared object is expected
 * to exist is given by the value of the VNCParameterRendererDirectory
 * parameter.
 *
 * Changes to this parameter after the viewer thread has started will have no
 * effect.
 *
 * An empty value indicates that the system library search method should be
 * used.
 *
 * A value of "." will use the current working directory.
 *
 * Default "" (system library search method).
 */
#define VNCParameterRendererDirectory "RendererDirectory"

#endif /* !defined(__VNCPARAMETERS_H__) */
